// ************************************************************
//
//                     Mini spectro Capteur AS7262
//
//   A utiliser avec l'interface graphique "minispectro.exe"
//
//            Stéphane LAURENT - Lycée Valin (17)
//
//*************************************************************


//*********************APPEL DES BIBLIOTHEQUES*********************************************************************

#include <Wire.h>                 //appel de la bibliotheque permettant de gérer les protocoles I2C

#include <LiquidCrystal_I2C.h>    //appel de la bibliotheque cristaux liquide pour l'ecran  

#include "AS726X.h"               //appel de la bibliotheque du capteur             

AS726X sensor;                    //creation de l'objet sensor de classe AS726X

LiquidCrystal_I2C lcd(0x27, 16, 2); //creation de l'objet lcd de classe LiquidCrystal_I2C

//******************* DEFINITION DES CONSTANTES *********************************************************************************************************

int intensiteSeuil = 150 ;     //Seuil d'intensité pour la LED

// definition des variables contenant des mesures
float intensities [6] ;   // intensite lumineuse mesuree par le capteur. Tableau contenant les valeurs aux 6 longueurs d'onde.
float blankIntensities [6]; // Tableau des intensite mesurees sur le blanc.
float absIntensities [6]; // Tableau des absorbances aux 6 longueurs d'onde.

//definition des pins hors ecran LCD
const int ledPin = 3 ;    //on se place sur un port PWM
int ledIntensity = 1 ;     //intensité de la LED à calibrer
bool ledOn = false ;      // etat de la LED, quand ledOn = false, la l'état de la LED est éteint

bool modeViolet = false ;   //booléens utilisés pour la fonction de changement de mode
bool modeBleu = false ;
bool modeVert = false ;
bool modeJaune = false ;
bool modeOrange = false ;
bool modeRouge = false ;

//******************* SETUP **************************************************************************************//

void setup() {

  lcd.init() ;               // initialisation du LCD
  lcd.backlight() ;          // active le rétroéclairage
  Serial.begin(9600) ;       // Mise en route du port Serie (communication avec l'ordinateur). On precise la vitesse de communication
  Serial.flush();            // On vide le tampon de réception par précaution
  pinMode(ledPin, OUTPUT) ;  // mode de la pin a laquelle est connectee la led en OUTPUT

  sensor.begin() ;           // mise en route du capteur
  sensor.setIntegrationTime(255) ;

  
//*******************************************************************************************************************//

  lcd.clear() ;
  switchOffLed() ;

  lcd.print("Initialisation...") ;  //affichage d'un message sur l'ecran

  ledCalibration() ;

  affichageInit();           // affichage initial lorsque le spectro est prêt

  Serial.println("B");      // message envoyé sur la liaison série pour l'informer que l'initialisation est terminée
}

//******************* LOOP s'execute en boucle en permanence ******************************************************************************************//

void loop() {

  if ( Serial.available() ) {
    String message = Serial.readString();  // lit la valeur envoyée et l’enregistre dans la variable "message" »

    if (message == "blanc") {
      takeBlank() ;                    // faire le blanc
      affichageInit();
    }
    
    if (message == "spectre") {
      lcd.clear() ;                    // on efface l'ecran LCD
      lcd.print("    Spectre") ;       // on affiche "Spectre d'absorption" sur l'ecran LCD
      lcd.setCursor(2, 1);
      lcd.print("d'absorption") ;
      mesureAbsorbances();             // faire le spectre d'absorbance
      printSerialAbs();
      affichageInit();
    }
    
    if (message.indexOf("0 nm") > 0) { // choix de la longueur d'onde

      choix_longueur_onde(message);
      affichageInit();
    }

    if (message == "absorbance") {
      lcd.clear() ;                    // on efface l'ecran LCD
      lcd.print("    Mesure de") ;     // on affiche "Spectre d'absorption" sur l'ecran LCD
      lcd.setCursor(2, 1);
      lcd.print("l'absorbance") ;
      mesureAbsorbances();            // mesure de l'absorbance
      printSimpleAbs();
      affichageInit();
    }
  }
  delay(100);
}

//******************* DEFINITION DES FONCTIONS UTILISEES ***************************************************************************************************//


void affichageInit() {              // Affichage initial lorsque le spectro est prêt
  lcd.clear() ;
  lcd.print("  Mini Spectro") ;
  lcd.setCursor(0, 1);              // mettre le curseur sur la deuxième ligne
  lcd.print("Lycee Valin (17)");
}
// choix de la longueur d'onde

void choix_longueur_onde(String longueur_onde) {

  modeViolet = false ;
  modeBleu = false ;
  modeVert = false ;
  modeJaune = false ;
  modeOrange = false ;
  modeRouge = false ;

  if (longueur_onde == "450 nm") {
    modeViolet = true;
  }
  if (longueur_onde == "500 nm") {
    modeBleu = true;
  }
  if (longueur_onde == "550 nm") {
    modeVert = true;
  }
  if (longueur_onde == "570 nm") {
    modeJaune = true;
  }
  if (longueur_onde == "600 nm") {
    modeOrange = true;
  }
  if (longueur_onde == "650 nm") {
    modeRouge = true;
  }

  lcd.clear() ;                    // on efface l'ecran LCD
  lcd.print(" Longueur d'onde") ;
  lcd.setCursor(5, 1);  // mettre le curseur à la colonne 6, ligne 2
  lcd.print(longueur_onde);
  delay(1000);

}

void mesureAbsorbances() {
  takeMeasure() ;
  computeAbsorbance(intensities, blankIntensities) ;
}

void printSimpleAbs() { // envoyer sur le port série l'absorbance à la longueur d'onde sélectionnée

  if (modeViolet == true) {

    Serial.println(int(absIntensities [0] * 1000));
  }
  if (modeBleu == true) {
    Serial.println(int(absIntensities [1] * 1000));
  }
  if (modeVert == true) {
    Serial.println(int(absIntensities [2] * 1000));
  }
  if (modeJaune == true) {
    Serial.println(int(absIntensities [3] * 1000));
  }
  if (modeOrange == true) {
    Serial.println(int(absIntensities [4] * 1000));
  }
  if (modeRouge == true) {
    Serial.println(int(absIntensities [5] * 1000));
  }
  lcd.clear() ;
  lcd.print(" Mesure terminee") ;
  delay(1000) ;
}

// ------------------------------------------------------------------------------------------------------------
// Envoyer les valeurs des absorbances sur le port série

void printSerialAbs() {
  for (int i = 0; i <= 4; i++) {
    Serial.print(int(absIntensities [i] * 1000)); // multiplier par 1000 pour envoyer uniquement des nombres entiers
    Serial.print(" "); // séparer les valeurs par un espace
  }
  Serial.println(int(absIntensities [5] * 1000));

  lcd.clear() ;
  lcd.print("Spectre termine") ;
  delay(1000) ;
}

//------------------------------------------------------------------------
//la fonction takeMeasure enregistre les intensites lumineuses mesurees pour chaque longueur d'onde

void takeMeasure() {

  switchOnLed();                // on allume la LED
  delay(1000) ;
  sensor.takeMeasurements() ;   // le capteur prend les mesures et les met en memoire
  switchOffLed() ;              // on eteint la LED
  //on affecte les valeurs de chaque longueur d'onde aux 6 variables associees
  intensities [0] = sensor.getCalibratedViolet() ;  // Le capteur va chercher les mesures mises en mémoire avec sensor.takeMeasurements(). La variable intensities[0] est affectée de cette valeur.
  intensities [1] = sensor.getCalibratedBlue() ;
  intensities [2] = sensor.getCalibratedGreen() ;
  intensities [3] = sensor.getCalibratedYellow() ;
  intensities [4] = sensor.getCalibratedOrange() ;
  intensities [5] = sensor.getCalibratedRed() ;
  delay(100);                  // on attend 100ms
}

//------------------------------------------------------------------------------------------------------------

// la fonction takeBlank permet de faire le blanc 

void takeBlank() {
  lcd.clear() ;                     // on efface l'ecran LCD
  lcd.print("Reglage du zero") ;    // on affiche "Réglage du zéro" sur l'ecran LCD
  delay(500) ;                      // on attend 0.5 s

  // On fait la mesure ici :
  takeMeasure();
  for (int k = 0; k < 6; k++) {
    blankIntensities [k] = intensities [k] ;  // on affecte a chaque case du tableau blankIntensities les valeurs d'intensite lumineuses mesurees a chaque longueur d'onde
    Serial.println(blankIntensities [k]);
  }

  lcd.setCursor(4, 1);
  lcd.print("termine !") ;
  switchOffLed() ;

  Serial.println("A"); // Envoyer "A" sur le port série pour indiquer que le blanc est terminé
  delay(1000) ;
}

///------------------------------------------------------------------------------------------------------------

//la fonction computeAbsorbance permet de calculer l'absorbance à partir des intensités luminseuses mesurées

void computeAbsorbance(float * newIntensities, float * refIntensities) {
  for (int k = 0; k < 6; k++) {
    absIntensities [k] = log10(refIntensities[k] / newIntensities[k])  ;        // on enregistre les valeurs calculées dans un tableau à 6 cases (pour 6 longueurs d'onde)
  }
}

// La fonction suivante permet d'allumer la LED blanche
void switchOnLed() {
  analogWrite(ledPin, ledIntensity) ; // on applique une tension de 5V  sur la pin de la LED, à une fréquence qui est égale à ledIntensity
  ledOn = true ;                     // l'état de la led passe à true : la LED est allumée
}

// La fonction suivante permet d'éteindre la LED blanche
void switchOffLed() {
  //on applique un potentiel de 0V sur la pin 7
  analogWrite(ledPin, 0) ;          // on applique une tension nulle surla pin de la LED
  ledOn = false ;                   // l'état de la LED passe à false : la LED est éteinte

}

void ledCalibration() {                     //Fonction qui permet de calibrer la LED au démarrage de l'appareil et donc de s'affranchir des différence qui peuvent exister entre les composants
  int violet = 0 ;
  ledIntensity = 0 ;                        //Cette valeur est proportionnellement liée au flux lumineux émis par la diode

  while (violet < intensiteSeuil) {                   //On cherche à atteindre une certaine intensité lumineuse lue par le capteur sur le canal Violet, ici 5000. Tant que cette valeur n'est pas atteinte au augmente ledIntensity
    analogWrite(ledPin, ledIntensity) ;
    delay(5) ;
    sensor.takeMeasurements() ;
    violet = sensor.getCalibratedViolet() ;
    ledIntensity ++ ;                                 //On incrémente l'intensité perçue par la LED
  }
}
